# Szyfrowanie z użyciem kryptografii klucza publicznego:
# https://www.nostarch.com/crackingcodes/ (na licencji BSD).

import sys, math

# Klucze publiczny i prywatny tego programu zostały
# utworzone przez program makePublicPrivateKeys.py
# Ten program musi być uruchomiony w katalogu zawierającym pliki kluczy.

SYMBOLS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890 !?.'

def main():
    # Sprawdzenie, czy ma być szyfrowana wiadomość do pliku, czy deszyfrowana
    # wiadomość z pliku.
    filename = 'encrypted_file.txt' # Plik, do którego zostaną zapisane dane lub z którego zostaną odczytane dane.
    mode = 'szyfrowanie'  # Przypisanie wartości 'szyfrowanie' lub 'deszyfrowanie'

    if mode == 'szyfrowanie':
        message = 'Journalists belong in the gutter because that is where the ruling classes throw their guilty secrets. Gerald Priestland. The Founding Fathers gave the free press the protection it must have to bare the secrets of government and inform the people. Hugo Black.'
        pubKeyFilename = 'al_sweigart_pubkey.txt'
        print('Szyfrowanie i zapis danych do pliku %s...' % (filename))
        encryptedText = encryptAndWriteToFile(filename, pubKeyFilename, message)

        print('Szyfrogram:')
        print(encryptedText)

    elif mode == 'deszyfrowanie':
        privKeyFilename = 'al_sweigart_privkey.txt'
        print('Odczyt danych z pliku %s i ich deszyfrowanie...' % (filename))
        decryptedText = readFromFileAndDecrypt(filename, privKeyFilename)

        print('Deszyfrowana wiadomość:')
        print(decryptedText)


def getBlocksFromText(message, blockSize):
    # Konwersja ciągu tekstowego wiadomości na listę bloków liczb całkowitych.
    for character in message:
        if character not in SYMBOLS:
            print('BŁĄD: Zbiór znaków nie zawiera znaku %s' % (character))
            sys.exit()
    blockInts = []
    for blockStart in range(0, len(message), blockSize):
        # Obliczenie liczby całkowitej bloku dla danego bloku tekstu.
        blockInt = 0
        for i in range(blockStart, min(blockStart + blockSize, len(message))):
            blockInt += (SYMBOLS.index(message[i])) * (len(SYMBOLS) ** (i % blockSize))
        blockInts.append(blockInt)
    return blockInts


def getTextFromBlocks(blockInts, messageLength, blockSize):
    # Konwertuje listę bloków liczb całkowitych na ciąg tekstowy pierwotnej wiadomości.
    # Długość pierwotnej wiadomości jest niezbędna do prawidłowej konwersji
    # ostatniego bloku liczby całkowitej.
    message = []
    for blockInt in blockInts:
        blockMessage = []
        for i in range(blockSize - 1, -1, -1):
            if len(message) + i < messageLength:
                # Deszyfrowanie ciągu tekstowego message dla 128 (lub innej wartości
                # przypisanej zmiennej blockSize) znaków z tego bloku liczby całkowitej.
                charIndex = blockInt // (len(SYMBOLS) ** i)
                blockInt = blockInt % (len(SYMBOLS) ** i)
                blockMessage.insert(0, SYMBOLS[charIndex])
        message.extend(blockMessage)
    return ''.join(message)


def encryptMessage(message, key, blockSize):
    # Konwertuje ciąg tekstowy message na listę liczb całkowitych bloków, a następnie
    # je szyfruje; w celu szyfrowania jest przekazywany klucz publiczny.
    encryptedBlocks = []
    n, e = key

    for block in getBlocksFromText(message, blockSize):
        # ciphertext = plaintext ^ e mod n
        encryptedBlocks.append(pow(block, e, n))
    return encryptedBlocks


def decryptMessage(encryptedBlocks, messageLength, key, blockSize):
    # Deszyfrowanie listy szyfrowanych bloków na postać ciągu tekstowego pierwotnej wiadomości.
    # Długość tego pierwotnego ciągu tekstowego jest wymagana, aby można było poprawnie
    # deszyfrować ostatni blok. Upewnij się, że został przekazany klucz PRYWATNY niezbędny do deszyfrowania.
    decryptedBlocks = []
    n, d = key
    for block in encryptedBlocks:
        # plaintext = ciphertext ^ d mod n
        decryptedBlocks.append(pow(block, d, n))
    return getTextFromBlocks(decryptedBlocks, messageLength, blockSize)


def readKeyFile(keyFilename):
    # Mając nazwę pliku zawierającego klucz publiczny lub prywatny,
    # funkcja zwraca klucz jako krotkę wartości (n,e) lub (n,d).
    fo = open(keyFilename)
    content = fo.read()
    fo.close()
    keySize, n, EorD = content.split(',')
    return (int(keySize), int(n), int(EorD))


def encryptAndWriteToFile(messageFilename, keyFilename, message, blockSize=None):
    # Użycie klucza odczytanego z pliku klucza, szyfrowanie wiadomości, a następnie zapisanie
    # jej w pliku; wartością zwrotną jest ciąg tekstowy szyfrogramu.
    keySize, n, e = readKeyFile(keyFilename)
    if blockSize == None:
        # Jeżeli wartość blockSize nie została podana, zostanie użyta największa dozwolona przez wielkości klucza i zbioru znaków.
        blockSize = int(math.log(2 ** keySize, len(SYMBOLS)))
    # Sprawdzenie, czy wielkość klucza jest wystarczająca dla wielkości bloku.
    if not (math.log(2 ** keySize, len(SYMBOLS)) >= blockSize):
        sys.exit('BŁĄD: Blok jest zbyt duży dla wybranych wielkości klucza i zbioru znaków. Czy na pewno wybrałeś prawidłowe pliki klucza i szyfrogramu?')
    # Zaszyfrowana wiadomość.
    encryptedBlocks = encryptMessage(message, (n, e), blockSize)

    # Konwersja ogromnych wartości w postaci liczb całkowitych na jeden ciąg tekstowy.
    for i in range(len(encryptedBlocks)):
        encryptedBlocks[i] = str(encryptedBlocks[i])
    encryptedContent = ','.join(encryptedBlocks)

    # Zapisanie zaszyfrowanego ciągu tekstowego w pliku danych wyjściowych.
    encryptedContent = '%s_%s_%s' % (len(message), blockSize, encryptedContent)
    fo = open(messageFilename, 'w')
    fo.write(encryptedContent)
    fo.close()
    # Wartością zwrotną jest zaszyfrowany ciąg tekstowy.
    return encryptedContent


def readFromFileAndDecrypt(messageFilename, keyFilename):
    # Użycie klucza pochodzącego z pliku klucza, odczytanie szyfrogramu z pliku,
    # a następnie jego deszyfrowanie; wartością zwrotną jest pierwotna wiadomość.
    keySize, n, d = readKeyFile(keyFilename)


    # Odczytanie z pliku wartości określającej długość wiadomości oraz ciągu tekstowego szyfrogramu.
    fo = open(messageFilename)
    content = fo.read()
    messageLength, blockSize, encryptedMessage = content.split('_')
    messageLength = int(messageLength)
    blockSize = int(blockSize)

    # Sprawdzenie, czy wielkość klucza jest wystarczająca dla wielkości bloku.
    if not (math.log(2 ** keySize, len(SYMBOLS)) >= blockSize):
        sys.exit('BŁĄD: Blok jest zbyt duży dla wybranych wielkości klucza i zbioru znaków. Czy na pewno wybrałeś prawidłowe pliki klucza i szyfrogramu?')

    # Konwersja szyfrogramu na wartości w postaci liczb całkowitych.
    encryptedBlocks = []
    for block in encryptedMessage.split(','):
        encryptedBlocks.append(int(block))

    # Deszyfrowanie ogromnych liczb całkowitych.
    return decryptMessage(encryptedBlocks, messageLength, (n, d), blockSize)


# Jeżeli program publicKeyCipher.p został uruchomiony (a nie zaimportowany
# jako moduł), należy wywołać funkcję main().
if __name__ == '__main__':
    main()
